﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace HIPS.Web.Components.Cache
{
    public static class ICacheProviderExtensions
    {
        /// <summary>
        /// Retrieve an item from the cache if it exists.
        /// Otherwise generate and, if the generated result is not null, add the item to the cache.
        /// Uses default expiry settings.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cache">The cache.</param>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="getItemCallback">Callback to a function to generate the item if it is not present in the cache. Null items will not be cached. If the callback should be cached for null results another value should be returned.</param>
        /// <returns>
        /// Item requested from the cache (or the generated value if it was not present).
        /// </returns>
        public static T GetOrSet<T>(this ICacheProvider cache, string cacheKey, Func<T> getItemCallback) where T : class
        {
            return cache.GetOrSet(cacheKey, getItemCallback, cache.GetDefaultSlidingExpiration(), cache.GetDefaultAbsoluteExpiration());
        }

        /// <summary>
        /// Retrieve an item from the cache if it exists.
        /// Otherwise, if the update condition is met, generate and, if the generated result is not null, add the item to the cache.
        /// Uses default expiry settings.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cache">The cache.</param>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="getItemCallback">Callback to a function to generate the item if it is not present in the cache. Null items will not be cached. If the callback should be cached for null results another value should be returned.</param>
        /// <param name="updateCondition">The update condition to apply against getItemCallback to determine if the value should be cached.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        /// <returns>
        /// Item requested from the cache (or the generated value if it was not present).
        /// </returns>
        public static T GetOrSetIf<T>(this ICacheProvider cache, string cacheKey, Func<T> getItemCallback, Func<T, bool> updateCondition, params string[] dependencyKeys) where T : class
        {
            return cache.GetOrSetIf(cacheKey, getItemCallback, updateCondition, cache.GetDefaultSlidingExpiration(), cache.GetDefaultAbsoluteExpiration(), dependencyKeys);
        }

        /// <summary>
        /// Retrieve an item from the cache if it exists.
        /// Otherwise generate and, if the generated result is not null, add the item to the cache.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cache">The cache.</param>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="getItemCallback">Callback to a function to generate the item if it is not present in the cache. Null items will not be cached. If the callback should be cached for null results another value should be returned.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        /// <returns>
        /// Item requested from the cache (or the generated value if it was not present).
        /// </returns>
        public static T GetOrSet<T>(this ICacheProvider cache, string cacheKey, Func<T> getItemCallback, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys) where T : class
        {
            return cache.GetOrSetIf(cacheKey, getItemCallback, null, slidingExpiration, absoluteExpiration, dependencyKeys);
        }

        /// <summary>
        /// Retrieve an item from the cache if it exists.
        /// Otherwise, if the update condition is met, generate and, if the generated result is not null, add the item to the cache.
        /// If the update condition is met add the item to the cache (if it is not null).
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cache">The cache.</param>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="getItemCallback">Callback to a function to generate the item if it is not present in the cache. Null items will not be cached. If the callback should be cached for null results another value should be returned.</param>
        /// <param name="updateCondition">The update condition to apply against getItemCallback to determine if the value should be cached.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        /// <returns>
        /// Item requested from the cache (or the generated value if it was not present).
        /// </returns>
        public static T GetOrSetIf<T>(this ICacheProvider cache, string cacheKey, Func<T> getItemCallback, Func<T, bool> updateCondition, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys) where T : class
        {
            // Return cached value if exists
            T item = cache.Get<T>(cacheKey);
            if (item != null)
            {
                return item;
            }

            // Double-checked locking (to prevent unnecessary invocations of getItemCallback)
            // See: http://stackoverflow.com/questions/39112/what-is-the-best-way-to-lock-cache-in-asp-net
            lock (cache.GetLock(cacheKey))
            {
                // Ensure cache wasn't updated while waiting for lock
                // Return cached value if exists
                item = cache.Get<T>(cacheKey);
                if (item != null)
                {
                    return item;
                }

                // Generate item
                item = getItemCallback();

                // Don't cache item if it's null (uncacheable)
                if (item == null)
                {
                    return null;
                }

                // Don't cache item if there's an updateCondiiton specified and not met
                if (updateCondition != null && !updateCondition(item))
                {
                    // Simply return the item in this case
                    return item;
                }

                // Add item to cache
                cache.Set(cacheKey, item, slidingExpiration, absoluteExpiration, dependencyKeys);
            }

            return item;
        }
    }
}
